/*	
 Copyright 2013 Adobe Systems Incorporated.  All rights reserved. 

Purpose- 
This file has the implementation of for Dreamweaver side utility functions which are exposed to extensions.
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global GenericLiveEditBridger,dw, DW_LIVEEDIT_CONSTANTS */

function ExtensionsBridgeFunctions(documentDOM) {
    'use strict';
    this.logDebugMsg("creating Extension Bridger");
    this.setDocumentDOM(documentDOM);
}

ExtensionsBridgeFunctions.prototype = new GenericLiveEditBridger();
ExtensionsBridgeFunctions.prototype.constructor = ExtensionsBridgeFunctions;
ExtensionsBridgeFunctions.prototype.baseClass = GenericLiveEditBridger.prototype.constructor;

/*
Function: getDWDocumentClassesAndIds - Gets the list of available classes and Ids for use in this document
Argument Obj Structure:
argObj.callback -> callback to be called after executing

Return Object : 
allSelectors - array of available selectors

Get User Dom Elements Classes and Id's for the passed element
*/
ExtensionsBridgeFunctions.prototype.getDWDocumentClassesAndIds = function (functionCallbackObj) {

    'use strict';
    var allSelectors = null;
    var theDOM = this.getDocumentDOM();
    if (theDOM) {
        allSelectors = theDOM.getSelectorsDefinedInStylesheet('id');
        if (allSelectors) {

            // Remove elements whose ids are being used in the DOM
            var nodeList = theDOM.getElementsByAttributeName('id');
            if (nodeList)
            {
                for (var i = 0; i < nodeList.length; i++)
                {
                    var nodeId = nodeList[i].getAttribute('id');
                    if(nodeId) {
                        var index = allSelectors.indexOf('#' + nodeId);
                        if (index !== -1) {
                            allSelectors.splice(index, 1);
                        }
                    }
                }
            }

            // concat class elements
            allSelectors = allSelectors.concat(theDOM.getSelectorsDefinedInStylesheet('class'));
        } else {
            // get class elements
            allSelectors = theDOM.getSelectorsDefinedInStylesheet('class');
        }
    } else {
        this.logDebugMsg("ERROR: getDWDocumentClassesAndIds: Can not find document DOM");
    }

    if (functionCallbackObj && functionCallbackObj.callback) {
        functionCallbackObj.callback(allSelectors);
    } else {
        this.logDebugMsg("ERROR: getDWDocumentClassesAndIds: Invalid callback provided");
    }
};

/*
Function: getUserDOMElementClassAndID
Argument Obj Structure:
argObj.uniqId -> dwid of element
argObj.callback -> callback to be called after executing

Return Object : 
newCallbackOb.idStr -> id of the element 
newCallbackOb.classStr ->  classes of the element

Get User Dom Elements Classes and Id's for the passed element
*/
ExtensionsBridgeFunctions.prototype.getDWDocumentElementClassAndID = function (argObj) {

    'use strict';
    this.logDebugMsg("getDWDocumentElementClassAndID from ExtensionsBridgeFunctions");

	var userDOMElement = null;
	var classStr;
	var idStr;
	if (argObj) {
		var uniqId = argObj.uniqId;
		if (uniqId && uniqId.length > 0) {
			userDOMElement = this.getElementByDWId(uniqId);
        }

		if (userDOMElement) {
			classStr = userDOMElement.getAttribute("class");
			idStr = userDOMElement.getAttribute("id");
		}
		var newCallbackObj = {};
		newCallbackObj.idStr = idStr;
		newCallbackObj.classStr = classStr;

		if (argObj.callback) {
			argObj.callback(newCallbackObj);
        }
	}
};

/*
Function: DWSMSetSelectionOnNode
Utility function to set selection on a node

Arguments : node

Return value : True for success, false for Failure 

Set Selection on a given node
*/
ExtensionsBridgeFunctions.prototype.DWSMSetSelectionOnNode = function (node) {
    
    if (!node) {
        this.logDebugMsg("node is not present");
    }
    var theDOM = this.getDocumentDOM();
    var offsets = theDOM.nodeToOffsets(node);
    theDOM.setSelection(offsets[0], offsets[1]);
    var curSelection = theDOM.getSelection();
    if (curSelection && (offsets[0] == curSelection[0]) && (offsets[1] == curSelection[1])) {
        return true;
    } else {
        var dwSelectNode = theDOM.getSelectedNode();
        if (dwSelectNode && node !== dwSelectNode) {
            theDOM.setSelectedNode(node, false, false, true, false);
            curSelection = theDOM.getSelection();
            if (curSelection && (offsets[0] == curSelection[0]) && (offsets[1] == curSelection[1])) {
                return true;
            }
        }   
    }
    return false;
};
/*
function:DWSMGetDWIdForAnchorTag
(Internal function on spider monkey side)
Arguments:argObj
    
    imgId - dwid of image under edit
    parentId - dwid ofParentNode of that image
    newText    - new text to keep in the pplace of old content at that offset.
    counter - how many nodes have been modified(to get corresponding node dwid's for tempid's)
    callback - callback function to be called

Description: 
    Update the link attribute and fetch dwid's for all the modfied nodes.

*/
ExtensionsBridgeFunctions.prototype.DWSMGetDWIdForAnchorTag = function (argObj) {
    'use strict';
    this.logDebugMsg("DWSMGetDWIdForAnchorTag from ExtensionsBridgeFunctions");

    if (!argObj) {
        this.logDebugMsg("DWSMGetDWIdForAnchorTag :callback Obj not present");
        return;
    }

    var theDOM = this.getDocumentDOM();
    if (!theDOM) {

        this.logDebugMsg("DWSMGetDWIdForAnchorTag : DOM is not available");
        return;
    }
    var imgNode = this.getElementByDWId(argObj.imgId);
    var parentNode = this.getElementByDWId(argObj.parentId);
	var actualParent = imgNode.parentNode;
    if (!imgNode || !parentNode) {
        this.logDebugMsg("DWSMGetDWIdForAnchorTag : nodes are invalid" + !imgNode + " " + !parentNode + argObj.imgId + "  " + argObj.parentId);
        return;
    }

	// Check parents are matching. In case of Editable Region, it won't match
	// because of ER comment node is treated as parent in DW side.
	if (parentNode !== actualParent && !this.isElementERComment(actualParent)) {
		throw DW_LIVEEDIT_CONSTANTS.ParentNotMatching;
	}

    var isSetSelectionSuccessful = this.DWSMSetSelectionOnNode(imgNode);    

    if (isSetSelectionSuccessful) {
        theDOM.insertHTML(argObj.newText, true, true);           
    } else {
        var offsets = theDOM.nodeToOffsets(imgNode);
		theDOM.setSelectionAndInsertHTML(offsets[0], offsets[1], argObj.newText, false, true, true);
    }
    var newIdObj = {};
    this.populateNewIDObjectInfo(parentNode, newIdObj); // populate new DWID's with repsect to temporary Id's
	
	// Check for Temp Id population error
	// When Image HUD removes/add link to image, It always cause structure change 
	// and excpets atleast one new ID and temp ID always starts from 0.
	// So check for '0' temp ID would make sure success.
	if (!newIdObj[0]) {
		throw DW_LIVEEDIT_CONSTANTS.ParentNotMatching;
	}
	
    // set selection explicitly to image under edit.(current selection is with Anchor tag)
    var image_new_id = newIdObj[argObj.img_callback_id]; // new data attribute Id image tag has got
    if (image_new_id) {
        imgNode = this.getElementByDWId(image_new_id); // image node
        if (imgNode) {
            this.DWSMSetSelectionOnNode(imgNode);
        }
    }
    newIdObj.parentId = argObj.parentId;
    newIdObj.callbackId = argObj.counter;
    argObj.callback(newIdObj);
};

/*
function:dwBrowseForFileURL
(Internal function on spider monkey side)
Arguments:argObj
        
        operation: select/open
        subop : name that will be shown at the top
        fileter: filters to be applied while showing the folder 
        callback: on successful/failure of execution callback to js layer.

Description: 
    Browse for file functionality for source and link.

*/
ExtensionsBridgeFunctions.prototype.dwBrowseForFileURL = function (argObj) {
    'use strict';
    if (!argObj) {
        this.logDebugMsg("dwBrowseForFileURL :callback Obj not present");
        return;
    }
    var imageFilter = false;
    if (argObj.filter.indexOf("png") !== -1) {
        imageFilter = true;
    }
    var path = dw.browseForFileURL(argObj.operation, argObj.subOp, imageFilter, false, argObj.filter);
    /* on Mac, the focus is to be explicitly set to the Live view */
	dw.setFocusToLiveView();
    if (argObj.callback) {
        argObj.callback(path);
    }
};

/*
function:rollBackFailedChanges
(Internal function on spider monkey side)
Arguments:none
Description: 
    Undo last change and clean up the redo events.
*/
ExtensionsBridgeFunctions.prototype.rollBackFailedChanges = function () {
	'use strict';
	this.logDebugMsg('ExtensionsBridgeFunctions.prototype.rollBackFailedChanges');

	var dom = this.getDocumentDOM();
	if (!dom) {
		return;
	}

	try {
		if (this.isDWTempIDLeftInDom(dom)) {
			this.logDebugMsg('isDWTempIDLeftInDom is TRUE - doing UNDO');
			var domsource = dom.source;
			if (domsource) {
				domsource.undo();
				dom.clearRedo();
				//domsource.clearRedo();
			}
		}
	} catch (e) {
		this.logDebugMsg('ERROR in rollBackFailedChanges : ' + e.message);
	}
};
